/***************************************************************************************/
/**                          Main window of the FG4 tester.
 *                          (c)2019-2020 Digiteq automotive
 ****************************************************************************************/
#include <vector>
#include <fstream>
#include <sstream>

#include <QString>
#include <QSettings>
#include <QMessageBox>
#include <QFileDialog>
#include <QtGui>
#include <QApplication>
#include <QMessageBox>
#include <QWidget>
#include <QtNetwork>
#include <QScrollBar>

#include "ui_FG4Window.h"
#include "FG4Window.h"
#include "AboutDialog.h"
#include "SynthParams.h"

#include "FG4_GUI_Version.h"

#ifdef _WINDOWS
  #include <Windows.h>
  #include <process.h>
#else
  #include <string.h>
  #include <poll.h> 
  #include <errno.h>  
#endif //_WINDOWS

#include "logfile.h"
#include "logmsg.h"
#include "logQt.h"

#include "xdma.h"
#include "BarAccess.h"

#include "stringa.h"
#include "raster.h"
#include "ras_prot.h"


extern "C" {
unsigned GetTickCount_ms(void);
}

#define SYNTH_HEIGHT	640
#define SYNTH_WIDTH	1280
#define SYNTH_FPS	30

///////////////////////////////////////////////////////////////////////////////////////////////


class Raster2D_8BitRGBAext: public Raster2D_8BitRGBA
{
public:
	Raster2D_8BitRGBAext(unsigned Size1D, unsigned Size2D, char *Blob);
	~Raster2D_8BitRGBAext();
};


Raster2D_8BitRGBAext::Raster2D_8BitRGBAext(unsigned NewSize1D, unsigned NewSize2D, char *Blob)
{
  if(Blob==NULL)
  {
    Size1D = Size2D = 0;
    return;
  }

  Data2D = (void **)malloc(sizeof(void *)*NewSize2D);
  if(Data2D==NULL)
  {
    Size1D = Size2D = 0;
    return;
  }

  for(unsigned i=0; i<NewSize2D; i++)
  {    
    Data2D[i] = Blob + 4*i*NewSize1D;
  }
  Size1D = NewSize1D;
  Size2D = NewSize2D;
}

Raster2D_8BitRGBAext::~Raster2D_8BitRGBAext()
{
  free(Data2D);
  Data1D = NULL;
  Data2D = NULL;
  Size1D = Size2D = 0;
}


#if QT_VERSION <0x050000
 #define CONN_TYPE	bool
#else
 #define CONN_TYPE	QMetaObject::Connection
#endif


extern LogFile g_TgtLogFile;
extern TgtQt   g_TgtLogQt;
LogFile TgtLogFile2(100000 /*filesize (bytes)*/, 1 /*maxfiles*/);

#define DEVICE_NAME "FG4 Test GUI"



///////////////////////////////////////////////////////////////////////////////////////////////

unsigned int __stdcall FG4_TestGui::FG4_JobThread(void *param)
{
FG4_TestGui *pWindow = (FG4_TestGui*)param;
const HudaqResourceInfo *HRI;

  if(param==NULL) return -1;

  if(pWindow->hDaq==NULL) return -2;
  HRI = HudaqGetDeviceResources(pWindow->hDaq);

  while(!pWindow->SigTerminateThread && pWindow->hDaq!=NULL)
  {
    HudaqWaitForIrq(pWindow->hDaq, 500, 1);			// Wait 100ms for possible IRQ
    pWindow->IrqLoop++;

    DWORD x = GetDword((size_t)HRI->MemResources[1].Base,0x2048);	// Unmasked IRQ requests.
    DWORD chnlIntPendReg = GetDword((size_t)HRI->MemResources[1].Base,0x204C);

    DWORD RegB4 = 0; 
    if(x & 1)
    {
      RegB4 = 1;
      pWindow->Frame0++;
      pWindow->Frame0ready();		 // poslat signal o pijet obrazku
    }
    if(x & 2)
    {
      RegB4 |= 2;
      pWindow->Frame1++;
      pWindow->Frame1ready();		 // poslat signal o pijet obrazku
    }
    if(RegB4 & 3)
    {
      StoreDword((size_t)HRI->MemResources[1].Base, 0x2008, RegB4 & 3);	// W1S - Unblock "User Interrupt Enable Mask"
      StoreDword((size_t)HRI->MemResources[0].Base, 0xB4, RegB4 & 3);	// Retrig frame grabbing.      
    }
    
  }

  return 0;
}


FG4_TestGui::FG4_TestGui(QWidget *parent): QMainWindow(parent), m_ui(new Ui::ControlDeskWindow)
{
   m_fileName = "";
   m_IsDirty = false;
   m_TimerRunning = false;
   hDaq = 0;
   LastBaseAddr = 0xFFFFFFFF;
   FileCounter = 0;
   FrameThread = NULL;
   SigTerminateThread = false;
   IrqLoop = Frame0 = Frame1 = 0;
   oldIrqLoop = oldFrame0 = oldFrame1 = 0;
   m_SynthHeight = SYNTH_HEIGHT;
   m_SynthWidth = SYNTH_WIDTH;
   m_FPS = SYNTH_FPS;
   m_HackLastPixel = true;

   m_ui->setupUi(this);

   pImg = new QImage(512, 400, QImage::Format_RGB32);
   for(unsigned x=0; x<256; x++)
     for(unsigned y=0; y<256; y++)
     {
       QRgb value;   
       value = qRgb(rand()%256, rand()%256, rand()%256); // 0xffbd9527
       pImg->setPixel(x, y, value);
     }   
   
   m_ui->Lbl_Image->setPixmap(QPixmap::fromImage(*pImg));
   m_ui->Lbl_Image->setVisible(true);

   
   m_ui->Tbr_main->setVisible(false);

	// Fill some pre-selected values

	// Load (set up) table

	// Update the gui user controls
   update();
   
	// Timer - periodic
   m_timer2 = new QTimer(this);
   connect(m_timer2, SIGNAL(timeout()), this, SLOT(slotStatusBar()));
   m_timer2->setInterval(1000);

   //connect(this, SIGNAL(Frame0ready()), this, SLOT(slotGetFrame()));

   m_timer = new QTimer(this);
   connect(m_timer, SIGNAL(timeout()), this, SLOT(slotGetFrame()));
   m_timer->setInterval(100);

   m_timerGen = new QTimer(this);
   connect(m_timerGen, SIGNAL(timeout()), this, SLOT(slotFeedFrame()));
   m_timerGen->setInterval(1000/30);

   m_IsAbortionRequested = false;
   
	/* Connect a target to display the logged messages to be used within the GUI */
   CONN_TYPE Result = QObject::connect(&g_TgtLogQt, SIGNAL(logMsgAdded(QString)), this, SLOT(slotLogMsgAdded(QString)));
   if(!Result)
            LogMessageQT(LOG_FATAL, tr("Cannot connect to logger slot.")).Print();

	// Run the full self-check
   QTimer::singleShot(500, this, SLOT(slotTriggered_SelfCheck()));

    //m_ui->centralwidget->setWindowTitle(DEVICE_NAME " Control Desk");
   this->setWindowTitle(DEVICE_NAME " FG4 Test GUI");
   StartupUDP();
}


FG4_TestGui::~FG4_TestGui()
{
  SigTerminateThread = true;
  if(m_timer) {delete m_timer; m_timer=NULL;}
  if(m_timer2) {delete m_timer2; m_timer2=NULL;}
  if(m_timerGen) {delete m_timerGen; m_timerGen=NULL;}
  if(pImg) {delete(pImg); pImg=NULL;}

  if(FrameThread!=NULL)
  {
    if(WaitForSingleObject(FrameThread, 5000) == WAIT_TIMEOUT) // Wait for thread to be finished WIN.
    {
      TerminateThread(FrameThread,-1000);	// Kill hanged thread.
    }
    CloseHandle(FrameThread);
    FrameThread = 0;
  }

  if(hDaq != 0)
  {
    HudaqCloseDevice(hDaq);
    hDaq = 0;
  }
  
  if(m_ui) {delete m_ui; m_ui=NULL;}
}


/** Event on close - request of confirmation */
void FG4_TestGui::closeEvent(QCloseEvent *event)
{
QString str;
int ActiveSockets = 0;

  if(m_timerGen && m_timerGen->isActive())
  {
    m_timerGen->stop();   
  }

  if(m_timer && m_timer->isActive())
  {
    if(QMessageBox::Yes != QMessageBox::question(this, tr("Confirmation"), 
             tr("Previous connection job was not finished.\nWould you like to abort it now?"), QMessageBox::Yes|QMessageBox::No))
    {
      event->ignore();
      return;    
    }
    m_timer->stop();
    if(m_timer2 && m_timer2->isActive()) m_timer2->stop();
    event->accept();
    return;
  }

  if(ActiveSockets<=0)
  {
    if(m_timer2 && m_timer2->isActive()) m_timer2->stop();
    event->accept();
    return;
  }

  event->accept(); 
}


/** Update window title, used in Save/Save As procedures */
void FG4_TestGui::updateWindowTitle(QString scenarioName)
{
  setWindowTitle(QString(scenarioName + tr(" - %0, V%1")).arg(VERSIONPRODUCTNAME).arg(VERSIONSTR));
}


/** Key press event */
void FG4_TestGui::keyPressEvent(QKeyEvent *keyEvent)
{
  switch(keyEvent->key())     // "Switch" here in case of future extension to more keys
  {
    case Qt::Key_F10:
        break;
  }
}


void FG4_TestGui::slotTriggered_SelfCheck(void)
{
    // Write message to log about start of check log
  LogMessageQT(LOG_INFO, tr("-------Self-check started.-------")).Print();
    /* It might be of an interest to know, how much free space is available on the drive, where $WORKSPACE is located.
       However, in the current version, it is only a nice-to-have feature. */
  LogMessageQT(LOG_INFO, tr("-------End of self-check summary-------.")).Print();
}



/// Show about dialog.
void FG4_TestGui::slotTriggered_About(void)
{
  AboutDialog *dialogAbout = new AboutDialog(this);
  dialogAbout->exec();
  delete dialogAbout;
}


/// Show Synth params dialog.
void FG4_TestGui::slotTriggered_SynthParams(void)
{
  SynthParams *dialogSynth = new SynthParams(this);
  dialogSynth->SetWidth(m_SynthWidth);
  dialogSynth->SetHeight(m_SynthHeight);
  dialogSynth->SetFPS(m_FPS);
  dialogSynth->SetHackLastPx(m_HackLastPixel);
  int result = dialogSynth->exec();
  if(result == 1)
  {
    m_SynthWidth = dialogSynth->GetWidth();
    m_SynthHeight = dialogSynth->GetHeight();
    m_FPS = dialogSynth->GetFPS();
    m_HackLastPixel = dialogSynth->GetHackLastPx();
  }
  delete dialogSynth;
}


/// Show user manual.
void FG4_TestGui::slotTriggered_UMA(void)
{
  QString DirPath = QApplication::applicationDirPath() + "/Doc/ControlDesk.pdf";
  QUrl pathUrl = QUrl::fromLocalFile(DirPath);
  if(!QDesktopServices::openUrl(pathUrl))
  {
    QMessageBox::warning(this, tr("Control Desk"), tr("The pdf file with User Manual cannot be found!"));
  }
}


/// This slot should stop all connections running.
void FG4_TestGui::slotStop(void)
{
  LogMessageQT(LOG_INFO, tr("slotStop clicked.")).Print();

  if(m_timer && m_timer->isActive())
  {
    m_timer->stop();  
  }

  if(m_timer2 && m_timer2->isActive()) m_timer2->stop();

  if(hDaq!=0)
  {
    const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);
    if(HRI!=NULL)
    {
      unsigned Mask = m_ui->MNU_DualPixel->isChecked() ? 0x200 : 0;
      StoreDword((size_t)HRI->MemResources[0].Base, 0x00, Mask|0x0);	// Retrig frame grabbing.
      StoreDword((size_t)HRI->MemResources[0].Base, 0x30, Mask|0x0);
    }
  }
}


void FG4_TestGui::slotDetach(void)
{
  LogMessageQT(LOG_INFO, tr("slotDetach clicked.")).Print();

  if(m_timer && m_timer->isActive())
  {
    if(QMessageBox::Yes != QMessageBox::question(this, tr("Confirmation"), 
             tr("Previous connection job was not finished.\nWould you like to abort it now?"), QMessageBox::Yes|QMessageBox::No))
    {      
      return;    
    }
    m_timer->stop(); 
  }

  if(m_timer2 && m_timer2->isActive()) m_timer2->stop();
}


/// Read configuration from external file.
void FG4_TestGui::slotConfiguration(void)
{
  QString conf_file;
  conf_file = QFileDialog::getOpenFileName(this,
                                            tr("Open configuration file"),
                                            QDir::currentPath(),
                                            tr("Configuration File (*.conf)"));
  
  conf_file = conf_file.trimmed();
  if(conf_file.isEmpty()) return;
  LoadConfiguration(conf_file);
}


/// Load configuration from a given file.
void FG4_TestGui::LoadConfiguration(const QString &conf_file)
{
  std::ifstream infile;
  infile.open(conf_file.toLocal8Bit().constData());

  if(infile.fail())
  {
    LogMessageQT(LOG_ERROR, tr("Cannot open configuration file: '%0'").arg(conf_file)).Print();
    return;
  }  
  
  infile.close();  
}


void FG4_TestGui::slotStart(void)
{
  if(m_timer) m_timer->stop();
  if(hDaq==0)
  {
    hDaq = HudaqOpenDevice("FG4",1,HudaqOpenNOINIT);
    if(hDaq==0)
    {
      QMessageBox::critical(this, QString("FG4 demo"), tr("No HUDAQ FG4 device found."));      
      return;
    }

    void *Pointer;
    unsigned __int64 DMA_MemRaw;
    unsigned BlockSz = 0;
    HudaqGetDMAinfo(hDaq,&DMA_MemRaw,&Pointer,&BlockSz);
    if(BlockSz>0 && BlockSz<0x800000)
    {
      HudaqReleaseDMAmem(hDaq);
      BlockSz = 0;
    }
    if(BlockSz==0)
    {
      HudaqAllocateDMAmem(hDaq,0x800000);
      HudaqGetDMAinfo(hDaq,&DMA_MemRaw,(void**)&Pointer,&BlockSz);
    }
    if(BlockSz<0x800000)
    {
      HudaqReleaseDMAmem(hDaq);
      QMessageBox::critical(this, QString("FG4 demo"), tr("Cannot allocate sufficient memory for DMA operation."));
      return;
    }
  }

  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);
  if(HRI!=NULL)
  {
    unsigned Mask = m_ui->MNU_DualPixel->isChecked() ? 0x200 : 0;
    StoreDword((size_t)HRI->MemResources[0].Base, 0x00, Mask|0x02);	// Retrig frame grabbing.
    StoreDword((size_t)HRI->MemResources[0].Base, 0x30, Mask|0x02);
  }
  
  if(m_timer)
  {
    LastBaseAddr = 0xFFFFFFFF;
    connect(m_timer, SIGNAL(timeout()), this, SLOT(slotGetFrame()));
    m_timer->setInterval(1000/30);
    m_timer->start();
  }
  if(m_timer2) m_timer2->start();
	
  if(FrameThread!=NULL)
  {		// Check whether thread is still living.
    if(WaitForSingleObject(FrameThread,1) == WAIT_OBJECT_0)
    {
      CloseHandle(FrameThread);
      FrameThread = NULL;      
    }
  }

  if(FrameThread==NULL)
  {
    DWORD tid;
    SigTerminateThread = false;
    FrameThread = (HANDLE)_beginthreadex(NULL,0,&FG4_JobThread,this,0,&tid);
  } 
}


void FG4_TestGui::StartOrAttach(void)
{
  if(m_timer && m_timer->isActive())
  {
    QMessageBox::warning(this, QString("ControlDesk"), tr("Previous connection job was not finished."));
    return;
  }

	// Send CANet configuration to all CANet servers, one by one.

  LogMessageQT(LOG_INFO, tr("Starting CANeT Network.")).Print();  
  if(m_timer2) m_timer2->stop();    
}


void FG4_TestGui::slotAttach(void)
{
  m_Attach = true;
  slotStart();
}


/** Generally, every slot should be logged. There must be an exception here, otherwise logging
    of this activity would lead to an infinite loop of logging actions. */
void FG4_TestGui::slotLogMsgAdded(QString MSG)
{ 
}


void FG4_TestGui::slotGetFrame(void)
{
unsigned __int64 DMA_MemRaw;
const HudaqResourceInfo *HRI;
int i;
QRgb *Pointer;
size_t BlockSz;
UINT32 BaseAddr;

  if(hDaq==0) return;

  if(m_TimerRunning) return;
  m_TimerRunning = true;

  HRI = HudaqGetDeviceResources(hDaq);
  if(HRI==NULL) 
  {
    m_TimerRunning = false;
    return;
  }

  i = HudaqGetDMAinfo(hDaq,&DMA_MemRaw,(void**)&Pointer,&BlockSz);
  if(i!=HUDAQSUCCESS || BlockSz<0x800000) return;

  unsigned Res = GetDword(HRI->MemResources[0].Base,0x8);
  if(pImg)
  {
    if((Res&0xFFFF)!=pImg->width() || (Res>>16)!=pImg->height())
    {
      delete(pImg);
      pImg = NULL;
    }
  }

  if(pImg==NULL)
  {
    pImg = new QImage(Res>>16, Res&0xFFFF, QImage::Format_RGB32);
  }

  if(pImg)
  {
    BaseAddr = GetDword(HRI->MemResources[0].Base,0x10);
    if(LastBaseAddr==BaseAddr)
    {
      m_TimerRunning = false;
      LogMessage(LOG_INFO, "No new frame ready %X.", BaseAddr).Print();
      return;
    }
    LastBaseAddr = BaseAddr;
    if((BaseAddr&0xFFFF0000)==0xFFFF0000)
    {		// No frame or invalid frame.
      m_TimerRunning = false;
      LogMessage(LOG_ERROR, "Wrong frame %X.", BaseAddr).Print();
      return;
    }    

    LogMessage(LOG_DEBUG, "Loading frame #%d, %ux%u.", FileCounter, (Res>>16), (Res&0xFFFF)).Print();
    Pointer = (QRgb*)DMA_Transfer0(hDaq, 0,  BaseAddr, 4*(Res>>16)*(Res&0xFFFF), 0);

    if(Pointer!=NULL)
    {
      for(unsigned y=0; y<pImg->height(); y++)
      {
        QRgb *rowData = (QRgb*)pImg->scanLine(y);    
        memcpy(rowData, Pointer+y*pImg->width(), pImg->width()*4);
      }
      if(m_timer->isActive())
          m_ui->Lbl_Image->setPixmap(QPixmap::fromImage(*pImg));
    }

    if(m_ui->Mnu_StoreBitmaps->isChecked())
    {
      string Filename;
      Filename.printf("Frame%u.bmp",FileCounter++);
      if(FileCounter>1000) FileCounter=0;

      Raster2D_8BitRGBAext *pRas = new Raster2D_8BitRGBAext(Res>>16, Res&0xFFFF, (char*)Pointer);
      Image img;
      img.AttachRaster(pRas); 
      SavePicture(Filename(),img);
    }
  }

  m_TimerRunning = false;
}


void FG4_TestGui::slotConnectStage2(void)
{
  if(m_TimerRunning) return;		// prevent timer retriggering.
  m_TimerRunning = true;

  if(m_timer) m_timer->stop();
  //Sleep(2000);
		// all servers have been configured and are ready for start  
  
  disconnect(m_timer, SIGNAL(timeout()), 0, 0);	//Disconnect everything connected to a specific signal  

  if(m_timer2) m_timer2->start();		// Start periodical pooling of servers
  m_TimerRunning = false;
}


void FG4_TestGui::slotStatusBar(void)
{
QString str = QString("Channel 1: %1 %2[fps]; Channel 2: %3 %4[fps]").arg(Frame0).arg(Frame0-oldFrame0).arg(Frame1).arg(Frame1-oldFrame1);
  m_ui->statusbar->showMessage(str);
  oldFrame0 = Frame0;
  oldFrame1 = Frame1;
}


/// Clear table with logs.
void FG4_TestGui::slotClearLog(void)
{  
}


/// Exit application.
void FG4_TestGui::slotExit(void)
{
  close();
  //m_ui->Txt_Status->
}


void FG4_TestGui::slotStartSynth(void)
{
const HudaqResourceInfo *HRI;

  m_timerGen->stop();
  m_FrameCounter = 0;
  m_ReceivedIRQ = 0;

  if(hDaq==0)
  {
    hDaq = HudaqOpenDevice("FG4",1,HudaqOpenNOINIT);
    if(hDaq==0)
    {
      QMessageBox::critical(this, QString("FG4 demo"), tr("No FG4 service interface found."));
      return;
    }
  }
  HRI = HudaqGetDeviceResources(hDaq);

  void *Pointer;
  unsigned __int64 DMA_MemRaw;
  unsigned BlockSz = 0;
  HudaqGetDMAinfo(hDaq,&DMA_MemRaw,&Pointer,&BlockSz);
  if(BlockSz>0 && BlockSz<0x100000)
  {
    HudaqReleaseDMAmem(hDaq);
    BlockSz = 0;
  }
  if(BlockSz==0)
  {
    HudaqAllocateDMAmem(hDaq,0x800000);				// Try to allocate 8MiB
    HudaqGetDMAinfo(hDaq,&DMA_MemRaw,(void**)&Pointer,&BlockSz);
    if(BlockSz==0)
    {
      HudaqAllocateDMAmem(hDaq,0x200000);			// Allocate 2MiB
      HudaqGetDMAinfo(hDaq,&DMA_MemRaw,(void**)&Pointer,&BlockSz);
      if(BlockSz==0)
      {
        HudaqAllocateDMAmem(hDaq,0x100000);			// Allocate 1MiB
        HudaqGetDMAinfo(hDaq,&DMA_MemRaw,(void**)&Pointer,&BlockSz);
      }
    }
  }
  if(BlockSz<0x100000)
  {
    HudaqReleaseDMAmem(hDaq);
    QMessageBox::critical(this, QString("FG4 demo"), tr("Cannot allocate sufficient memory for DMA operation."));
    return;
  }

  StoreDword((size_t)HRI->MemResources[0].Base, OUT1_RESOLUTION_RW, (m_SynthWidth<<16) |  m_SynthHeight);	// Set resolution 1280 x 640
  StoreDword((size_t)HRI->MemResources[0].Base, OUT1_FRAMERATE_RW, (LONG)(FRAME_MEAS_CLK / m_FPS)); // Set framerate 30fps
  StoreDword((size_t)HRI->MemResources[0].Base, OUT1_CONFIG_RW, (0x2|0x8));		// Set PC queue 0.

  OutBuff1 = 0xFFFFFFFF;
  FeedFrameRunning = false;
  slotFeedFrame();
  m_timerGen->start();
}


void FG4_TestGui::slotEndSynth(void)
{
  bool TimerStarted = m_timerGen->isActive();
  m_timerGen->stop();
  if(hDaq==0) return;
  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);
  if(HRI==NULL) return;
  StoreDword((size_t)HRI->MemResources[0].Base, OUT1_CONFIG_RW, 0);
  FeedFrameRunning = false;
  if(TimerStarted)
    QMessageBox::information(this, QString("FG4 demo"), tr("Frames sent %1\nIRQs received %2").arg(m_FrameCounter).arg(m_ReceivedIRQ));
  m_ReceivedIRQ = m_FrameCounter = 0;
}


void FG4_TestGui::slotFeedFrame(void)
{
char *Pointer;
unsigned __int64 DMA_MemRaw;
unsigned BlockSz = 0;
unsigned char j = m_FrameCounter % 255;

  if(FeedFrameRunning) return;
 
  if(hDaq==0) return;
  const HudaqResourceInfo *HRI = HudaqGetDeviceResources(hDaq);
  if(HRI==NULL) return;
  HudaqGetDMAinfo(hDaq,&DMA_MemRaw,(void**)&Pointer,&BlockSz);

  FeedFrameRunning = true;
  if(OutBuff1 >= 0xFFFF0000)
  {
    OutBuff1 = GetDword((size_t)HRI->MemResources[0].Base, OUT1_NEW_BUF_ADDRESS_R);
    if(OutBuff1 >= 0xFFFF0000)
    {
      FeedFrameRunning = false;
      return;
    }
  }

  //Sleep(1);

  DWORD len = m_SynthWidth * m_SynthHeight * 4;
  while(len > 0)
  {
    unsigned len2 = len;
    if(len2 > BlockSz-sizeof(DMA_DESCRIPTOR))
        len2 = BlockSz-sizeof(DMA_DESCRIPTOR);
    unsigned DWORD *Pointer2 = (DWORD*)(Pointer + sizeof(DMA_DESCRIPTOR));

    for(unsigned i=0; i<len2; i+=4)
    {
      DWORD DW = (DWORD)j | ((DWORD)j<<8) | ((DWORD)j<<16);
      //DWORD DW = FrameCounter; //rand();
      //DW = 0xFFFFFF;
      if(j & 1) DW=0;
      *Pointer2 = DW;      
      /*Pointer[i+sizeof(DMA_DESCRIPTOR)] = rand() % 255;
      Pointer[i+sizeof(DMA_DESCRIPTOR)+1] = rand() % 255;
      Pointer[i+sizeof(DMA_DESCRIPTOR)+2] = rand() % 255;
      Pointer[i+sizeof(DMA_DESCRIPTOR)+3] = 0;*/
      j++;
      Pointer2++;
    }

    if(len==len2 && m_HackLastPixel)	// last fragment
    {
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-12] = 0;
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-11] = 255;
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-10] = m_FrameCounter % 255;
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-9] = 0;

      Pointer[sizeof(DMA_DESCRIPTOR)+len2-8] = 0;
      
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-4] = 255;
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-3] = m_FrameCounter % 255;
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-2] = 0;
      Pointer[sizeof(DMA_DESCRIPTOR)+len2-1] = 0;
      //memset(Pointer+sizeof(DMA_DESCRIPTOR)+len2-8, 0, 8);
    }

    unsigned CountIRQ = (unsigned)HudaqGetParameter(hDaq,0,HudaqIrqNUMCOUNTER);
    DMA_TransferH2C0(hDaq,0,OutBuff1, len2, 0);
    m_ReceivedIRQ += (unsigned)HudaqGetParameter(hDaq,0,HudaqIrqNUMCOUNTER) - CountIRQ;
    OutBuff1 += len2;					// advance remote addr
    len -= len2;
  }
  //Sleep(1);
  OutBuff1 = GetDword((size_t)HRI->MemResources[0].Base, OUT1_NEW_BUF_ADDRESS_R);
  m_FrameCounter++;
  FeedFrameRunning = false;
}
