/* GraphBuilder.cpp
 *   This demo demonstrates how to connect Direct Show components to a graph.
 *   This code connects FG4 renderer to FG4grabber video source.
 *   (c)2021-2024 Digiteq Automotive  written by Jaroslav Fojtik	*/
#include <Windows.h>
#include <conio.h>

#include <uuids.h>
#include <strmif.h>
#include <control.h>

#include <comdef.h>

#include "../../shared/FG4iface.h"


_COM_SMARTPTR_TYPEDEF(IGraphBuilder, __uuidof(IGraphBuilder));
_COM_SMARTPTR_TYPEDEF(IMediaControl, __uuidof(IMediaControl));
_COM_SMARTPTR_TYPEDEF(IAMStreamConfig, __uuidof(IAMStreamConfig));
_COM_SMARTPTR_TYPEDEF(ICaptureGraphBuilder2, __uuidof(ICaptureGraphBuilder2));

_COM_SMARTPTR_TYPEDEF(ICreateDevEnum, __uuidof(ICreateDevEnum));
_COM_SMARTPTR_TYPEDEF(IEnumMoniker, __uuidof(IEnumMoniker));
_COM_SMARTPTR_TYPEDEF(IMoniker, __uuidof(IMoniker));
_COM_SMARTPTR_TYPEDEF(IBaseFilter, __uuidof(IBaseFilter));
_COM_SMARTPTR_TYPEDEF(IFG4KsproxyConfig, __uuidof(IFG4KsproxyConfig));
_COM_SMARTPTR_TYPEDEF(IFG4KsproxyTransmitConfig, __uuidof(IFG4KsproxyTransmitConfig));
_COM_SMARTPTR_TYPEDEF(IPin, __uuidof(IPin));
_COM_SMARTPTR_TYPEDEF(IEnumPins, __uuidof(IEnumPins));


IMediaControlPtr    graph;		///< containder for camera-grabber stuff
ICaptureGraphBuilder2Ptr capbuilder;


void LocateFg4Camera(IBaseFilterPtr &BFilterIn, IBaseFilterPtr &BFilterOut, int DEVICE_ORDER=0)
{
HRESULT hResult;
    // Locate our capture filter.
    // First, create the device enumerator object

  //ICreateDevEnum *piCreateDevEnum;
  //hResult = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER, __uuidof(piCreateDevEnum), reinterpret_cast<void**>(&piCreateDevEnum));

  ICreateDevEnumPtr piCreateDevEnum;
  hResult = piCreateDevEnum.CreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER);

  if(SUCCEEDED(hResult))
  {              
    int Xorder = DEVICE_ORDER;
        // Get class enumerator for WDM Streaming Capture Devices category
    IEnumMonikerPtr piEnumMoniker;

    hResult = piCreateDevEnum->CreateClassEnumerator(AM_KSCATEGORY_CAPTURE, &piEnumMoniker, 0);
    //piCreateDevEnum->Release();

    if(SUCCEEDED(hResult))
        hResult = piEnumMoniker->Reset();

    if(SUCCEEDED(hResult))
    {				// Enumerate KS devices
      ULONG cFetched;
      IMonikerPtr piMoniker;
      while((hResult = piEnumMoniker->Next(1, &piMoniker, &cFetched)) == S_OK)
      {
        IBaseFilterPtr piFilter;

#ifdef _DEBUG
                // Printout devices we find (ignore any errors)
        LPOLESTR pwszMonikerName;
        if(SUCCEEDED(piMoniker->GetDisplayName(NULL, NULL, &pwszMonikerName)))
        {
          printf("Found device: %S\n", pwszMonikerName);
          CoTaskMemFree(pwszMonikerName);
        }
#endif  // _DEBUG

                // The device we look for is identified by supporting FG4KsproxySampleConfig.
                // Do check this we need to instantiate the filter object and then query it for
                // FG4KsproxySampleConfig.
                // This is not the fastest method.
                // We could rely on filter's name to avoid creating the filter
        hResult = piMoniker->BindToObject(NULL, NULL, __uuidof(piFilter), reinterpret_cast<void**>(&piFilter));
        if(SUCCEEDED(hResult))
        {
	  IFG4KsproxyConfigPtr piConfig;
          hResult = piFilter->QueryInterface(__uuidof(piConfig), reinterpret_cast<void**>(&piConfig));
          
          if(FAILED(hResult))
              piFilter.Detach()->Release();
	  else
	  {
	    if(Xorder > 0)
	    {
	      Xorder--;
	      piFilter.Detach()->Release();
	      piConfig.Detach()->Release();
	      hResult = E_FAIL;
            }
            else
            {
	      BFilterIn = piFilter;
	      piFilter.Detach()->Release();
	      piConfig.Detach()->Release();;
	      break;
            }
          }
        }

        piMoniker.Detach()->Release();
        if(SUCCEEDED(hResult)) break;
      }
    }

    piEnumMoniker.Detach()->Release();


        // Get class enumerator for WDM Streaming Renderer Devices category
    Xorder = 0;    
    hResult = piCreateDevEnum->CreateClassEnumerator(AM_KSCATEGORY_RENDER, &piEnumMoniker, 0);
    piCreateDevEnum.Detach()->Release();			// no longer needed.

    if(SUCCEEDED(hResult))
        hResult = piEnumMoniker->Reset();

    if(SUCCEEDED(hResult))
    {				// Enumerate KS devices
      ULONG cFetched;
      IMonikerPtr piMoniker;
      while((hResult = piEnumMoniker->Next(1, &piMoniker, &cFetched)) == S_OK)
      {
        IBaseFilterPtr piFilter;

#ifdef _DEBUG
                // Printout devices we find (ignore any errors)
        LPOLESTR pwszMonikerName;
        if(SUCCEEDED(piMoniker->GetDisplayName( NULL, NULL, &pwszMonikerName)))
        {
          printf("Found device: %S\n", pwszMonikerName);
          CoTaskMemFree(pwszMonikerName);
        }
#endif  // _DEBUG

                // The device we look for is identified by supporting FG4KsproxySampleConfig.
                // Do check this we need to instantiate the filter object and then query it for
                // FG4KsproxySampleConfig.
                // This is not the fastest method.
                // We could rely on filter's name to avoid creating the filter
        hResult = piMoniker->BindToObject(NULL, NULL, __uuidof(piFilter), reinterpret_cast<void**>(&piFilter));
        if(SUCCEEDED(hResult))
        {
	  IFG4KsproxyTransmitConfigPtr poConfig;
          hResult = piFilter->QueryInterface(__uuidof(poConfig), reinterpret_cast<void**>(&poConfig));
          
          if(FAILED(hResult))
              piFilter.Detach()->Release();
	  else
	  {
	    LPOLESTR pwszMonikerName;
            hResult = piMoniker->GetDisplayName( NULL, NULL, &pwszMonikerName);
            wchar_t *DevName = SUCCEEDED(hResult)?pwszMonikerName:NULL;
	    if(DevName!=NULL)
	    {
	      DevName = wcsstr(pwszMonikerName,L"}\\");
	      if(DevName) DevName+=2;
            }
	    printf("\nDevice transmitter #%d found %S", Xorder, (DevName!=NULL)?DevName:L"");
	    if(SUCCEEDED(hResult)) CoTaskMemFree(pwszMonikerName);
	    Xorder++;

	    BFilterOut = piFilter;
	    piFilter.Detach()->Release();
	    poConfig.Detach()->Release();
	    break;
          }
        }

        piMoniker.Detach()->Release();
      }
    }

    piEnumMoniker.Detach()->Release();
  }
}



void Help(void)
{
  printf("\nH	Show help");
  printf("\nP	Pause graph");
  printf("\nR	Run graph");
  printf("\nI	Stop graph - sometimes hangs, known problem");
  printf("\nS	Stop graph - should be safe");
  printf("\nX	Exit");
}


int UseGraph(void)
{
HRESULT hr;

///////////////////////////////////////////////////////

  IBaseFilterPtr camera_base;
  IBaseFilterPtr renderer_base;
  LocateFg4Camera(camera_base, renderer_base);

  if(camera_base == NULL)
  {
     printf("\nCannot find FG4 camera");
     return -2;
  }

  if(renderer_base == NULL)
  {
     printf("\nCannot find FG4 renderer");
     return -3;
  }

///////////////////////////////////////////////////////

  // create the filter graph and necessary interfaces  
  graph.CreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC);
  if (graph==NULL)
  {
    printf("\nFailed to create filter graph");
    return -4;
  }
  IGraphBuilderPtr graphbuilder(graph);
  if (graphbuilder==NULL)
  {
    printf("\nFailed to get GraphBuilder interface for the filter graph");
    return -5;
  }

  // create the capture graph builder
  capbuilder.CreateInstance(CLSID_CaptureGraphBuilder2, NULL, CLSCTX_INPROC);
  if (capbuilder==NULL)
  {
    printf("\nFailed to create capture graph builder");
    return -6;
  }

  // attach the filter graph to the capture graph builder
  hr = capbuilder->SetFiltergraph(graphbuilder);
  if (FAILED(hr))
  {
    printf("\nFailed to attach filter graph to the capture graph builder");
    return -7;
  }

 // add the camera filter to the filter graph  
  hr = graphbuilder->AddFilter(camera_base, L"Camera");
  if(FAILED(hr))
  {
    printf("\nFailed to add camera filter to filter graph");
    return -8;
  }

 // add the camera filter to the filter graph  
  hr = graphbuilder->AddFilter(renderer_base, L"Renderer");
  if(FAILED(hr))
  {
    printf("\nFailed to add renderer filter to filter graph");
    return -9;
  }

  IPinPtr CamPIN;
  IPinPtr RenderPIN;
  IEnumPinsPtr pEnum;

  hr = camera_base->EnumPins(&pEnum);
  hr = pEnum->Next(1,&CamPIN,NULL);
  if(FAILED(hr))
  {
    printf("\nCannot obtain output pin from camera");
    return -10;
  }

  pEnum.Detach()->Release();

  hr = renderer_base->EnumPins(&pEnum);
  hr = pEnum->Next(1,&RenderPIN,NULL);
  if(FAILED(hr))
  {
    printf("\nCannot obtain output pin from renderer");
    return -11;
  }

  pEnum.Detach()->Release();


  hr = graphbuilder->Connect(RenderPIN,CamPIN);
  if(FAILED(hr))
  {
    printf("\nCannot connect renderer with grabber, error %Xh.",hr);
    return -12;
  }

  char key = 0; 
  Help();
  while(key!='X')
  {
    if(!kbhit())
    {
      Sleep(10);
      continue;
    }
    key = toupper(getch());
      
    if(key=='H')
    {
      Help();
    }

    if(key=='I')
    {
      printf("\nPressed I ... Stopping graph");
      hr = renderer_base->Stop();
      if(FAILED(hr))
          printf("\nCannot stop renderrer, error %Xh", hr);
      else
          printf("; renderer stopped");
      hr = camera_base->Stop();
      if(FAILED(hr))
          printf("\nCannot stop camera, error %Xh", hr);
      else
          printf("; camera stopped");
    }

    if(key=='1')
    {
      printf("\nPressed 1 ... Stopping renderer only");
      hr = renderer_base->Stop();
      if(FAILED(hr))
          printf("\nCannot stop renderrer, error %Xh", hr);
      else
          printf("; renderer stopped");
    }
    if(key=='2')
    {
      printf("\nPressed 2 ... Stopping camera only");
      hr = camera_base->Stop();
      if(FAILED(hr))
          printf("\nCannot stop camera, error %Xh", hr);
      else
          printf("; camera stopped");
    }

    if(key=='S')
    {
      printf("\nPressed S ... Safe/Reverse stopping graph");
      hr = camera_base->Stop();
      if(FAILED(hr))
          printf("\nCannot stop camera, error %Xh", hr);
      else
          printf("; camera stopped");
      hr = renderer_base->Stop();
      if(FAILED(hr))
          printf("\nCannot stop renderrer, error %Xh", hr);
      else
          printf("; renderer stopped");
    }

    if(key=='P')
    {
      printf("\nPressed P ... Pausing graph");
      hr = renderer_base->Pause();
      if(FAILED(hr))
         printf("\nCannot pause renderrer, error %Xh", hr);
      hr = camera_base->Pause();
      if(FAILED(hr))
         printf("\nCannot pause camera, error %Xh", hr);
    }
    if(key=='R')
    {
      printf("\nPressed R ... Starting graph");
      hr = renderer_base->Run(0);
      if(FAILED(hr))
         printf("\nCannot start renderer, error %Xh", hr);
      hr = camera_base->Run(0);
      if(FAILED(hr))
         printf("\nCannot start camera, error %Xh", hr);
    }
  }
  printf("\nPressed X ... Application exit");

return 0;
}


int main(void)
{
HRESULT hr;

  hr = CoInitializeEx(NULL, COINIT_MULTITHREADED|COINIT_DISABLE_OLE1DDE);
  if(FAILED(hr))
  {
    return -1;
  }

 int ret = UseGraph();

  if(graph!=NULL)
     graph.Detach()->Release();

  if(capbuilder!=NULL)
    capbuilder.Detach()->Release();

 CoUninitialize();
return ret;
}
