//------------------------------------------------------------------------------
// File: fFrameBuffer.cpp
//
// Desc: DirectShow sample code - implementation of filter behaviors
//       for the bouncing ball source filter.  For more information,
//       refer to Ball.cpp.
//
// Copyright (c) Microsoft Corporation.  All rights reserved.
//------------------------------------------------------------------------------

#include "stdafx.h"
#include <streams.h>
#include <olectl.h>
#include <initguid.h>
#include "FrameBuffer.h"
#include "fFrameBuffer.h"


#ifdef _MSC_VER
 #pragma warning(disable:4710)  // 'function': function not inlined (optimzation)
#endif


// Setup data
const REGPINTYPES MediaTypes[1] = {
  {&MEDIATYPE_Video, &MEDIASUBTYPE_RGB32},	// 0 return a 32bit mediatype
};


typedef enum {
    BLACK = 0,
    WHITE,
    YELLOW,
    CYAN,
    GREEN,
    MAGENTA,
    RED,
    BLUE,
    GREY,

    MAX_COLOR
} COLOR;

const unsigned char ColorsARGB[MAX_COLOR][4] = {
    {0, 0, 0},          // BLACK
    {255, 255, 255},    // WHITE
    {0, 255, 255},      // YELLOW
    {255, 255, 0},      // CYAN
    {0, 255, 0},        // GREEN
    {255, 0, 255},      // MAGENTA
    {0, 0, 255},        // RED
    {255, 0, 0},        // BLUE
    {128, 128, 128}     // GREY
};



////////////////////////////////////////////////////////////////////////
//
// Exported entry points for registration and unregistration 
// (in this case they only call through to default implementations).
//
////////////////////////////////////////////////////////////////////////


// CreateInstance
// The only allowed way to create Bouncing balls!
CUnknown * WINAPI CFrameBuffer::CreateInstance(LPUNKNOWN lpunk, HRESULT *phr)
{
    ASSERT(phr);

    CUnknown *punk = new CFrameBuffer(lpunk, phr);
    if(punk == NULL)
    {
        if(phr)
            *phr = E_OUTOFMEMORY;
    }
    return punk;
} // CreateInstance


// Constructor
// Initialise a CFrameBuffStream object so that we have a pin.
CFrameBuffer::CFrameBuffer(LPUNKNOWN lpunk, HRESULT *phr): CSource(NAME("Frame buffer sample"), lpunk, CLSID_BouncingBall)
{
  ASSERT(phr);
  CAutoLock cAutoLock(&m_cStateLock);
  
  m_paStreams = (CSourceStream **)new CFrameBuffStream*[1];
  if(m_paStreams == NULL)
  {
    if(phr)
        *phr = E_OUTOFMEMORY;

    return;
  }

  m_paStreams[0] =  m_CFrameBuffStream = new CFrameBuffStream(phr, this, L"A Frame buffer sample!");
  if(m_paStreams[0] == NULL)
  {
    if(phr)
        *phr = E_OUTOFMEMORY;

    return;
  }
} // (Constructor)


/////////////IFeederProxySampleConfig//////////////////////////////////////////


HRESULT CFrameBuffer::GetWidth(long* plWidth)
{
  if(plWidth==NULL) return E_POINTER;
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  *plWidth = m_CFrameBuffStream->m_iImageWidth;
  return S_OK;
}


HRESULT CFrameBuffer::SetWidth(long lWidth)
{
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  m_CFrameBuffStream->m_iImageWidth = lWidth;
  return S_OK;
}


HRESULT CFrameBuffer::GetHeight(long* plHeight)
{
  if(plHeight==NULL) return E_POINTER;
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  *plHeight = m_CFrameBuffStream->m_iImageHeight;
  return S_OK;
}


HRESULT CFrameBuffer::SetHeight(long lHeight)
{
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  m_CFrameBuffStream->m_iImageHeight = lHeight;
  return S_OK;
}


HRESULT CFrameBuffer::GetRepeatTime_ms(long* plRepeatTime_ms)
{
  if(plRepeatTime_ms==NULL) return E_POINTER;
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  *plRepeatTime_ms = m_CFrameBuffStream->m_iRepeatTime;
  return S_OK;
}


HRESULT CFrameBuffer::SetRepeatTime_ms(long lRepeatTime_ms)
{
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  m_CFrameBuffStream->m_iRepeatTime = lRepeatTime_ms;
  return S_OK;
}


HRESULT CFrameBuffer::GetBallSize(long* plBallSize)
{
  if(plBallSize==NULL) return E_POINTER;
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  *plBallSize = m_CFrameBuffStream->m_iBallSize;
  return S_OK;
}


HRESULT CFrameBuffer::SetBallSize(long lBallSize)
{
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  m_CFrameBuffStream->m_iBallSize = lBallSize;
  return S_OK;
}


/// This method allows to feed frame into running camera.
/// Please note that the video flow should have already agreed frame rate.
/// When application miss several frames, synthetic feed video is generated.
/// @param[in]	BufPtr	Pointer to a video frame.
/// @param[in]	BufSize	Frame block size.
HRESULT CFrameBuffer::FeedExternalBuffer(unsigned const char *BufPtr, unsigned BufSize)
{
  if(m_State!=CFrameBuffStream::CMD_RUN)  return E_FAIL;
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  if(BufPtr==NULL || BufSize==0) return E_POINTER;
  return m_CFrameBuffStream->FeedExternalBuffer(BufPtr,BufSize);
}


HRESULT CFrameBuffer::GetSafetyTimer(long *plTimer)
{
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  if(plTimer==NULL) return E_POINTER;
  *plTimer = m_CFrameBuffStream->m_SafetyTimer;
  return S_OK;
}


HRESULT CFrameBuffer::SetSafetyTimer(long lTimer)
{
  if(m_CFrameBuffStream==NULL) return E_UNEXPECTED;
  m_CFrameBuffStream->m_SafetyTimer = lTimer;
  return S_OK;
}


/////////////IAMStreamConfig//////////////////////////////////////////


HRESULT CFrameBuffer::GetFormat(AM_MEDIA_TYPE **ppmt)
{
  if(m_CFrameBuffStream==NULL) return E_FAIL;
  return m_CFrameBuffStream->GetFormat(ppmt);
}


HRESULT CFrameBuffer::SetFormat(AM_MEDIA_TYPE *pmt)
{
  if(m_CFrameBuffStream==NULL) return E_FAIL;
  return m_CFrameBuffStream->SetFormat(pmt);
}


HRESULT CFrameBuffer::GetNumberOfCapabilities(int *piCount, int *piSize)
{
  if(piCount==NULL || piSize==NULL) return E_POINTER;
  *piCount = sizeof(MediaTypes) / sizeof(REGPINTYPES);
  *piSize = sizeof(VIDEO_STREAM_CONFIG_CAPS);
return S_OK;
}


HRESULT CFrameBuffer::GetStreamCaps(int iIndex, AM_MEDIA_TYPE **ppmt, BYTE *pSCC)
{
  if(m_CFrameBuffStream==NULL) return E_FAIL;
  return m_CFrameBuffStream->GetStreamCaps(iIndex, ppmt, pSCC);
}


///////////// IMediaFilter /////////////

HRESULT CFrameBuffer::Run(REFERENCE_TIME tStart)
{
  if(m_CFrameBuffStream!=NULL)
      m_CFrameBuffStream->Run();
  return CBaseFilter::Run(tStart);
}


 HRESULT CFrameBuffer::Pause(void)
{
  //if(m_CFrameBuffStream!=NULL)
  //    m_CFrameBuffStream->Pause();	// Populating Pause() hanghs application :( !
  return CBaseFilter::Pause();
}


HRESULT CFrameBuffer::Stop(void)
{
  if(m_CFrameBuffStream!=NULL)
      m_CFrameBuffStream->Stop();
  return CBaseFilter::Stop();
}


///////////////////////////////////////////////////////////////////////////////////

// Constructor
CFrameBuffStream::CFrameBuffStream(HRESULT *phr, CFrameBuffer *pParent, LPCWSTR pPinName):
    CSourceStream(NAME("Fake camera frame buffer"),phr, pParent, pPinName),
    m_iImageWidth(320),
    m_iImageHeight(240),
    m_iBallSize(25),
    m_iDefaultRepeatTime(40),
    LastExtFrame_ms(0),
    m_SafetyTimer(5)
{
    ASSERT(phr);
    CAutoLock cAutoLock(&m_cSharedState);

    m_Ball = new CBall(m_iImageWidth, m_iImageHeight, m_iBallSize);
    if(m_Ball == NULL)
    {
      if(phr)
          *phr = E_OUTOFMEMORY;
    }

    m_iRepeatTime = m_iDefaultRepeatTime;
} // (Constructor)


// Destructor
CFrameBuffStream::~CFrameBuffStream()
{
    CAutoLock cAutoLock(&m_cSharedState);
    if(m_Ball)
        delete m_Ball;
} // (Destructor)


/// This method is intended to inject frame from external application.
HRESULT CFrameBuffStream::FeedExternalBuffer(unsigned const char *BufPtr, unsigned BufSize)
{
//Command com;
  if(BufPtr==NULL) return E_POINTER;
  if(BufSize==0) return E_INVALIDARG;
  //if(CheckRequest(&com)) return E_FAIL;
  //if(com != CMD_RUN) return E_FAIL;

  LastExtFrame_ms = GetTickCount_ms();

  IMediaSample *pSample;
  HRESULT hr = GetDeliveryBuffer(&pSample,NULL,NULL,0);
  if(FAILED(hr)) return hr;
  if(pSample==NULL) return E_POINTER;

  BYTE *pData;
  long lDataLen;
  pSample->GetPointer(&pData);
  if(pData==NULL) return E_POINTER;
  lDataLen = pSample->GetSize();
  if(lDataLen<=0)
  {
    pSample->Release();
    return E_FAIL;
  }

  if(BufSize<lDataLen) lDataLen=BufSize;
  memcpy(pData, BufPtr, lDataLen);

  hr = Deliver(pSample);
  pSample->Release();
  LastFrame_ms = LastExtFrame_ms;
return hr;
}


// DoBufferProcessingLoop
// Grabs a buffer and calls the users processing function.
// Overridable, so that different delivery styles can be catered for.
HRESULT CFrameBuffStream::DoBufferProcessingLoop(void)
{
Command com;

  OnThreadStartPlay();

  do {
	while (!CheckRequest(&com)) {

		// Block standard feed when external feed is used.
            const unsigned CurrentTick = GetTickCount_ms();
            if((CurrentTick-LastExtFrame_ms) <= m_SafetyTimer*m_iRepeatTime)
            {
              Sleep(m_iRepeatTime);
              continue;
            }

	    IMediaSample *pSample;

	    HRESULT hr = GetDeliveryBuffer(&pSample,NULL,NULL,0);
	    if (FAILED(hr) ||
	       pSample==NULL) {			// FIX by JFO, NULL sometimes occurs.
                Sleep(1);
		continue;	// go round again. Perhaps the error will go away
			    // or the allocator is decommited & we will be asked to
			    // exit soon.
	    }

	    // Virtual function user will override.
	    hr = FillBuffer(pSample);

	    if (hr == S_OK) {
		hr = Deliver(pSample);
                pSample->Release();

                // downstream filter returns S_FALSE if it wants us to
                // stop or an error if it's reporting an error.
                if(hr != S_OK)
                {
                  DbgLog((LOG_TRACE, 2, TEXT("Deliver() returned %08x; stopping"), hr));
                  return S_OK;
                }

	    } else if (hr == S_FALSE) {
                // derived class wants us to stop pushing data
		pSample->Release();
		DeliverEndOfStream();
		return S_OK;
	    } else {
                // derived class encountered an error
                pSample->Release();
		DbgLog((LOG_ERROR, 1, TEXT("Error %08lX from FillBuffer!!!"), hr));
                DeliverEndOfStream();
                m_pFilter->NotifyEvent(EC_ERRORABORT, hr, 0);
                return hr;
	    }

            // all paths release the sample
	}

        // For all commands sent to us there must be a Reply call!

	if (com == CMD_RUN || com == CMD_PAUSE) {
	    Reply(NOERROR);
	} else if (com != CMD_STOP) {
	    Reply((DWORD) E_UNEXPECTED);
	    DbgLog((LOG_ERROR, 1, TEXT("Unexpected command!!!")));
	}
    } while (com != CMD_STOP);

    return S_FALSE;
}


// FillBuffer
// Plots a ball into the supplied video buffer
HRESULT CFrameBuffStream::FillBuffer(IMediaSample *pms)
{
  CheckPointer(pms,E_POINTER);
  ASSERT(m_Ball);

  BYTE *pData;
  long lDataLen;

  pms->GetPointer(&pData);
  lDataLen = pms->GetSize();

  //ZeroMemory(pData, lDataLen);
	// This is more fancy to feed color bars.
  {
    BYTE *ptr;
    ptr = pData + m_iPixelSize * m_iImageWidth;
    if(ptr > pData+lDataLen) ptr=pData+lDataLen;
    long i = 0;
    while(ptr>pData)
    {
      ptr -= m_iPixelSize;
      memcpy(ptr, ColorsARGB[(i++ *MAX_COLOR)/m_iImageWidth], 4);
    }
    ptr = pData + 4*m_iImageWidth;
    i = ptr - pData;
    while(2*i <= lDataLen)
    {
      memcpy(ptr, pData, i);
      ptr += i;
      i = ptr - pData;
    }
    if(2*i > lDataLen)
    {
      i = lDataLen - i;
      if(i>0) memcpy(ptr, pData, i);
    }
  }

  {
    CAutoLock cAutoLockShared(&m_cSharedState);

        // If we haven't just cleared the buffer delete the old
        // ball and move the ball on
    if(lDataLen >= m_iPixelSize * m_iImageHeight * m_iImageWidth)
    {
      m_Ball->MoveBall(m_rtSampleTime - (LONG)m_iRepeatTime);
      m_Ball->PlotBall(pData, m_BallPixel, m_iPixelSize);
    }

        // The current time is the sample's start
    CRefTime rtStart = m_rtSampleTime;

        // Increment to find the finish time
    m_rtSampleTime += (LONG)m_iRepeatTime;

    pms->SetTime((REFERENCE_TIME *) &rtStart,(REFERENCE_TIME *) &m_rtSampleTime);
  }

  pms->SetSyncPoint(TRUE);

  LastFrame_ms += m_iRepeatTime;
  unsigned CurrentTick = GetTickCount_ms();
  if(LastFrame_ms - CurrentTick < (unsigned)0x8000000)
  {
    unsigned Time2Sleep = LastFrame_ms - CurrentTick;	// Unsigned substract wraps correctly
    if(Time2Sleep > m_iRepeatTime) Time2Sleep=m_iRepeatTime;
    Sleep(Time2Sleep);
  }

  return NOERROR;
} // FillBuffer


// Notify
//
// Alter the repeat rate according to quality management messages sent from
// the downstream filter (often the renderer).  Wind it up or down according
// to the flooding level - also skip forward if we are notified of Late-ness
STDMETHODIMP CFrameBuffStream::Notify(IBaseFilter * pSender, Quality q)
{
    // Adjust the repeat rate.
    if(q.Proportion<=0)
    {
        m_iRepeatTime = 1000;        // We don't go slower than 1 per second
    }
    else
    {
        m_iRepeatTime = m_iRepeatTime*1000 / q.Proportion;
        if(m_iRepeatTime>1000)
        {
            m_iRepeatTime = 1000;    // We don't go slower than 1 per second
        }
        else if(m_iRepeatTime<10)
        {
            m_iRepeatTime = 10;      // We don't go faster than 100/sec
        }
    }

    // skip forwards
    if(q.Late > 0)
        m_rtSampleTime += q.Late;

    return NOERROR;

} // Notify


const CMediaType *CFrameBuffStream::GetCurrentMediaType(void) const
{
  return &m_mt;
}


// GetMediaType
//
// I _prefer_ 5 formats - 8, 16 (*2), 24 or 32 bits per pixel and
// I will suggest these with an image size of 320x240. However
// I can accept any image size which gives me some space to bounce.
//
// A bit of fun:
//      32 bit see yellow
//
// Prefered types should be ordered by quality, zero as highest quality
// Therefore iPosition =
// 0    return a 32bit mediatype
// (iPosition > 0 is invalid)
HRESULT CFrameBuffStream::GetMediaType(int iPosition, CMediaType *pmt)
{
    CheckPointer(pmt,E_POINTER);

    CAutoLock cAutoLock(m_pFilter->pStateLock());
    if(iPosition < 0)
    {
        return E_INVALIDARG;
    }

    // Have we run off the end of types?
    if(iPosition > 0)
    {
        return VFW_S_NO_MORE_ITEMS;
    }

    VIDEOINFO *pvi = (VIDEOINFO *) pmt->AllocFormatBuffer(sizeof(VIDEOINFO));
    if(NULL == pvi)
        return(E_OUTOFMEMORY);

    ZeroMemory(pvi, sizeof(VIDEOINFO));

    switch(iPosition)
    {
        case 0:
        {    
            // Return our highest quality 32bit format

            // since we use RGB888 (the default for 32 bit), there is
            // no reason to use BI_BITFIELDS to specify the RGB
            // masks. Also, not everything supports BI_BITFIELDS

            //SetPaletteEntries(Yellow);
            pvi->bmiHeader.biCompression = BI_RGB;
            pvi->bmiHeader.biBitCount    = 32;
            break;
        }
        default: return E_INVALIDARG;
    }

    // (Adjust the parameters common to all formats...)

	// put the optimal palette in place
/*    for(int i=0; i<iPALETTE_COLORS; i++)
    {
        pvi->TrueColorInfo.bmiColors[i].rgbRed      = m_Palette[i].peRed;
        pvi->TrueColorInfo.bmiColors[i].rgbBlue     = m_Palette[i].peBlue;
        pvi->TrueColorInfo.bmiColors[i].rgbGreen    = m_Palette[i].peGreen;
        pvi->TrueColorInfo.bmiColors[i].rgbReserved = 0;
    } */

    pvi->bmiHeader.biSize       = sizeof(BITMAPINFOHEADER);
    pvi->bmiHeader.biWidth      = m_iImageWidth;
    pvi->bmiHeader.biHeight     = m_iImageHeight;
    pvi->bmiHeader.biPlanes     = 1;
    pvi->bmiHeader.biSizeImage  = GetBitmapSize(&pvi->bmiHeader);
    pvi->bmiHeader.biClrImportant = 0;

    SetRectEmpty(&(pvi->rcSource)); // we want the whole image area rendered.
    SetRectEmpty(&(pvi->rcTarget)); // no particular destination rectangle

    pmt->SetType(&MEDIATYPE_Video);
    pmt->SetFormatType(&FORMAT_VideoInfo);
    pmt->SetTemporalCompression(FALSE);

    // Work out the GUID for the subtype from the header info.
    const GUID SubTypeGUID = GetBitmapSubtype(&pvi->bmiHeader);
    pmt->SetSubtype(&SubTypeGUID);
    pmt->SetSampleSize(pvi->bmiHeader.biSizeImage);

    return NOERROR;

} // GetMediaType


// CheckMediaType
// We will accept 8, 16, 24 or 32 bit video formats, in any
// image size that gives room to bounce.
// Returns E_INVALIDARG if the mediatype is not acceptable
//
HRESULT CFrameBuffStream::CheckMediaType(const CMediaType *pMediaType)
{
    CheckPointer(pMediaType,E_POINTER);

    if((*(pMediaType->Type()) != MEDIATYPE_Video) ||   // we only output video
       !(pMediaType->IsFixedSize()))                   // in fixed size samples
    {                                                  
        return E_INVALIDARG;
    }

    // Check for the subtypes we support
    const GUID *SubType = pMediaType->Subtype();
    if (SubType == NULL)
        return E_INVALIDARG;

    if(*SubType != MEDIASUBTYPE_RGB32)	// Only RGB32 is supported.
    {
        return E_INVALIDARG;
    }

    // Get the format area of the media type
    VIDEOINFO *pvi = (VIDEOINFO *) pMediaType->Format();

    if(pvi == NULL)
        return E_INVALIDARG;

    // Check the image size. As my default ball is 10 pixels big
    // look for at least a 20x20 image. This is an arbitary size constraint,
    // but it avoids balls that are bigger than the picture...

    if((pvi->bmiHeader.biWidth < 20) || ( abs(pvi->bmiHeader.biHeight) < 20))
    {
        return E_INVALIDARG;
    }

    // Check if the image width & height have changed
    if(pvi->bmiHeader.biWidth != m_iImageWidth ||
       abs(pvi->bmiHeader.biHeight) != m_iImageHeight)
    {
        // If the image width/height is changed, fail CheckMediaType() to force
        // the renderer to resize the image.
        return E_INVALIDARG;
    }

    return S_OK;  // This format is acceptable.

} // CheckMediaType


// DecideBufferSize
//
// This will always be called after the format has been sucessfully
// negotiated. So we have a look at m_mt to see what size image we agreed.
// Then we can ask for buffers of the correct size to contain them.
HRESULT CFrameBuffStream::DecideBufferSize(IMemAllocator *pAlloc, ALLOCATOR_PROPERTIES *pProperties)
{
    CheckPointer(pAlloc,E_POINTER);
    CheckPointer(pProperties,E_POINTER);

    CAutoLock cAutoLock(m_pFilter->pStateLock());
    HRESULT hr = NOERROR;

    VIDEOINFO *pvi = (VIDEOINFO *) m_mt.Format();
    pProperties->cBuffers = 1;
    pProperties->cbBuffer = pvi->bmiHeader.biSizeImage;

    ASSERT(pProperties->cbBuffer);

    // Ask the allocator to reserve us some sample memory, NOTE the function
    // can succeed (that is return NOERROR) but still not have allocated the
    // memory that we requested, so we must check we got whatever we wanted

    ALLOCATOR_PROPERTIES Actual;
    hr = pAlloc->SetProperties(pProperties,&Actual);
    if(FAILED(hr))
    {
        return hr;
    }

    // Is this allocator unsuitable

    if(Actual.cbBuffer < pProperties->cbBuffer)
    {
        return E_FAIL;
    }

    // Make sure that we have only 1 buffer (we erase the ball in the
    // old buffer to save having to zero a 200k+ buffer every time
    // we draw a frame)

    ASSERT(Actual.cBuffers == 1);
    return NOERROR;

} // DecideBufferSize


// SetMediaType
// Called when a media type is agreed between filters
HRESULT CFrameBuffStream::SetMediaType(const CMediaType *pMediaType)
{
    CAutoLock cAutoLock(m_pFilter->pStateLock());

    // Pass the call up to my base class

    HRESULT hr = CSourceStream::SetMediaType(pMediaType);

    if(SUCCEEDED(hr))
    {
        VIDEOINFO * pvi = (VIDEOINFO *) m_mt.Format();
        if (pvi == NULL)
            return E_UNEXPECTED;

	m_iRepeatTime = pvi->AvgTimePerFrame / 10000;

        switch(pvi->bmiHeader.biBitCount)
        {
            case 32:    // Make a yellow pixel

                m_BallPixel[0] = 0x0;
                m_BallPixel[1] = 0xA0;
                m_BallPixel[2] = 0xA0;
                m_BallPixel[3] = 0x00;
                m_iPixelSize   = 4;
                //SetPaletteEntries(Yellow);
                break;

            default:
                // We should never agree any other pixel sizes
                ASSERT(FALSE);
                break;
        }

        CBall *pNewBall = new CBall(pvi->bmiHeader.biWidth, abs(pvi->bmiHeader.biHeight), m_iBallSize);

        if(pNewBall)
        {
            delete m_Ball;
            m_Ball = pNewBall;
        }
        else
            hr = E_OUTOFMEMORY;

        return NOERROR;
    } 

    return hr;

} // SetMediaType


// As we go active reset the stream time to zero
HRESULT CFrameBuffStream::OnThreadCreate()
{
  CAutoLock cAutoLockShared(&m_cSharedState);
  m_rtSampleTime = 0;

    // we need to also reset the repeat time in case the system
    // clock is turned off after m_iRepeatTime gets very big
  if(m_iRepeatTime == 0)
      m_iRepeatTime = m_iDefaultRepeatTime;

  LastFrame_ms = GetTickCount_ms();

  return NOERROR;
} // OnThreadCreate


STDMETHODIMP CFrameBuffStream::NonDelegatingQueryInterface(REFIID riid, void ** ppv)
{
  if(riid == IID_IPin) {
      return GetInterface((IPin *) this, ppv);
  } else if (riid == IID_IQualityControl) {
      return GetInterface((IQualityControl *) this, ppv);
  } else if (riid == IID_IAMStreamConfig) {
      return GetInterface((IAMStreamConfig *) this, ppv);
  } else if (riid == IID_IKsPropertySet) {
      return GetInterface((IKsPropertySet *) this, ppv);
  } else {		// "31efac30-515c-11d0-a9aa-00aa0061be93" IKsPropertySet;
      return CUnknown::NonDelegatingQueryInterface(riid, ppv);
  }
}


//////////IAMStreamConfig///////////////////////////////////////////////


HRESULT CFrameBuffStream::GetFormat(AM_MEDIA_TYPE **ppmt)
{
  if(ppmt==NULL) return E_POINTER;

  const CMediaType *pCurrentMt = GetCurrentMediaType();

  *ppmt = (AM_MEDIA_TYPE*)CoTaskMemAlloc(sizeof(AM_MEDIA_TYPE));
  memcpy(*ppmt, (AM_MEDIA_TYPE*)pCurrentMt, sizeof(AM_MEDIA_TYPE));
  if((*ppmt)->pbFormat!=NULL && (*ppmt)->cbFormat>0)
  {
    (*ppmt)->pbFormat = (BYTE*)CoTaskMemAlloc(pCurrentMt->cbFormat);
    memcpy((*ppmt)->pbFormat, pCurrentMt->pbFormat, pCurrentMt->cbFormat);
    //VIDEOINFO *pvi = (VIDEOINFO *)(*ppmt)->pbFormat;
  }

return S_OK;
}


HRESULT CFrameBuffStream::SetFormat(AM_MEDIA_TYPE *pmt)
{
  if(pmt==NULL) return E_POINTER;
 
  for(int i=0; i<sizeof(MediaTypes)/sizeof(REGPINTYPES); i++)
  {
    if(*MediaTypes[i].clsMajorType==pmt->majortype &&
       *MediaTypes[i].clsMinorType==pmt->subtype)
    {
      CMediaType mt;
      HRESULT hr = GetMediaType(i,&mt);
      if(FAILED(hr)) return hr;
      mt = *pmt;
      return SetMediaType(&mt);
    }
  }
  return E_INVALIDARG;
}


HRESULT CFrameBuffStream::GetNumberOfCapabilities(int *piCount, int *piSize)
{
  if(piCount==NULL || piSize==NULL) return E_POINTER;
  *piCount = sizeof(MediaTypes) / sizeof(REGPINTYPES);
  *piSize = sizeof(VIDEO_STREAM_CONFIG_CAPS);
return S_OK;
}


HRESULT CFrameBuffStream::GetStreamCaps(int iIndex, AM_MEDIA_TYPE **ppmt, BYTE *pSCC)
{
  if(ppmt==NULL) return E_POINTER;

  CMediaType mt;
  HRESULT hr = GetMediaType(iIndex,&mt);
  if(FAILED(hr)) return hr;

  *ppmt = (AM_MEDIA_TYPE*)CoTaskMemAlloc(sizeof(AM_MEDIA_TYPE));
  memcpy(*ppmt, (AM_MEDIA_TYPE*)&mt, sizeof(AM_MEDIA_TYPE));
  if((*ppmt)->pbFormat!=NULL && (*ppmt)->cbFormat>=sizeof(VIDEOINFOHEADER))
  {
    (*ppmt)->pbFormat = (BYTE*)CoTaskMemAlloc(mt.cbFormat);
    memcpy((*ppmt)->pbFormat, mt.pbFormat, mt.cbFormat);
    VIDEOINFOHEADER *pvi = (VIDEOINFOHEADER *)(*ppmt)->pbFormat;
    if(pvi->AvgTimePerFrame==0)
      pvi->AvgTimePerFrame = m_iDefaultRepeatTime * 10000;	// 60Hz in 100ns intervals.
    if(pvi->dwBitRate==0)
      pvi->dwBitRate = pvi->bmiHeader.biSizeImage * 8 * 60;
  }

  if(pSCC!=NULL)
  {
    VIDEO_STREAM_CONFIG_CAPS *pScc = (VIDEO_STREAM_CONFIG_CAPS *)pSCC;
    pScc->guid = mt.formattype;
    pScc->VideoStandard = 0;
    pScc->InputSize.cx = pScc->MaxCroppingSize.cx = pScc->MinCroppingSize.cx = m_iImageWidth;
    pScc->InputSize.cy = pScc->MaxCroppingSize.cy = pScc->MinCroppingSize.cy = m_iImageHeight;
    pScc->CropGranularityX = pScc->CropGranularityY = 1;
    pScc->CropAlignX = pScc->CropAlignY = 1;
    pScc->MaxOutputSize.cx = pScc->MinOutputSize.cx = m_iImageWidth;
    pScc->MaxOutputSize.cy = pScc->MinOutputSize.cy = m_iImageHeight;
    pScc->OutputGranularityX = pScc->OutputGranularityY = 1;
    pScc->StretchTapsX = pScc->StretchTapsY = pScc->ShrinkTapsX = pScc->ShrinkTapsY = 0;
    pScc->MaxFrameInterval = 10000000;		// 1Hz
    pScc->MinFrameInterval = 10000000 / 60;	// 60Hz
    pScc->MinBitsPerSecond = m_iImageWidth * m_iImageHeight;
    switch(iIndex)
    {
      case 0: pScc->MinBitsPerSecond *= 4*8; break;	// return a 32bit mediatype
/*
      case 1: pScc->MinBitsPerSecond *= 3*8; break;	// return a 24bit mediatype
      case 2: pScc->MinBitsPerSecond *= 2*8; break;	// return 16bit RGB565
      case 3: pScc->MinBitsPerSecond *= 2*8; break;	// return a 16bit mediatype (rgb555)
      case 4: pScc->MinBitsPerSecond *= 8; break;	// return 8 bit palettised format
*/
    }
    pScc->MaxBitsPerSecond = pScc->MinBitsPerSecond * 60;
  }

return S_OK;
}


////////////////// IKsPropertySet//////////////////////////

HRESULT CFrameBuffStream::Get(REFGUID guidPropSet, DWORD dwPropID, LPVOID  pInstanceData, DWORD cbInstanceData, LPVOID pPropData, DWORD cbPropData, DWORD *pcbReturned)
{
  if(guidPropSet==AMPROPSETID_Pin)
  {
    switch(dwPropID)
    {
      case AMPROPERTY_PIN_CATEGORY: if(cbPropData<sizeof(GUID)) return E_INVALIDARG;
				    if(pPropData==NULL) return E_POINTER;
				    memcpy(pPropData, &PIN_CATEGORY_CAPTURE, sizeof(GUID));
				    if(pcbReturned) *pcbReturned = sizeof(GUID);
				    return S_OK;
    }
    return E_PROP_ID_UNSUPPORTED;
  }
  return E_PROP_SET_UNSUPPORTED;
}


HRESULT CFrameBuffStream::QuerySupported(REFGUID guidPropSet, DWORD dwPropID, DWORD *pTypeSupport)
{
  if(guidPropSet==AMPROPSETID_Pin)
  {
    if(dwPropID==AMPROPERTY_PIN_CATEGORY) return S_OK;
    return E_PROP_ID_UNSUPPORTED;
  }
  return E_PROP_SET_UNSUPPORTED;
}


HRESULT CFrameBuffStream::Set(REFGUID guidPropSet, DWORD dwPropID, LPVOID pInstanceData, DWORD cbInstanceData, LPVOID pPropData, DWORD cbPropData)
{
  return E_NOTIMPL;
}


/////////////////////////////////////////


HRESULT CFrameBuffStream::Run()
{
  LastFrame_ms = GetTickCount_ms();
  return CSourceStream::Run();
}